package com.ho.crypto.test3.encryption.aes;

import java.security.Key;
import java.security.Security;

import javax.crypto.Cipher;
import javax.crypto.KeyGenerator;
import javax.crypto.SecretKey;
import javax.crypto.spec.SecretKeySpec;

import org.apache.commons.lang.RandomStringUtils;


/*
 * Encryption with "AES/ECB/PKCS5Padding" ALGO on 128bits
 * https://docs.oracle.com/javase/7/docs/api/javax/crypto/Cipher.html
 * Every implementation of the Java platform is required to support the following standard Cipher transformations with the keysizes in parentheses:
		AES/CBC/NoPadding (128)
		AES/CBC/PKCS5Padding (128)
		AES/ECB/NoPadding (128)
		AES/ECB/PKCS5Padding (128)
		DES/CBC/NoPadding (56)
		DES/CBC/PKCS5Padding (56)
		DES/ECB/NoPadding (56)
		DES/ECB/PKCS5Padding (56)
		DESede/CBC/NoPadding (168)
		DESede/CBC/PKCS5Padding (168)
		DESede/ECB/NoPadding (168)
		DESede/ECB/PKCS5Padding (168)
		RSA/ECB/PKCS1Padding (1024, 2048)
		RSA/ECB/OAEPWithSHA-1AndMGF1Padding (1024, 2048)
		RSA/ECB/OAEPWithSHA-256AndMGF1Padding (1024, 2048)
 */
public class EncryptionAesEcbPkcs5Padding {
    private static final String ALGORITHM = "AES/ECB/PKCS5Padding";
    private static final String UNICODE_FORMAT  = "UTF-8";
    private static final int AES_KEYLENGTH = 128;
    public static enum INTERNAL_KEY_GENERATION_MODE {
    	RANDOM_BYTES_NOWRAPPED_KEY, GENERATED_NOWRAPPED_KEY, GENERATED_WRAPPED_KEY, 
    };
    
    /**
     * Constructor
     */
    public EncryptionAesEcbPkcs5Padding(){
    }

    /**
	 * Encrypt PDF file
	 * @param keyValue : secret key
     * @param data
     * @return
     * @throws Exception
     */
    public static byte[] encrypt(byte[] keyValue, byte[] data) throws Exception {
    	SecretKeySpec keyspec = new SecretKeySpec(keyValue, "AES");
        Cipher c = Cipher.getInstance(ALGORITHM);
        c.init(Cipher.ENCRYPT_MODE, keyspec);
        byte[] encVal = c.doFinal(data);
        return encVal;
    }

    /**
	 * Decrypt PDF file
	 * @param keyValue : secret key
     * @param data
     * @return
     * @throws Exception
     */
    public static byte[] decrypt(byte[] keyValue, byte[] encryptedData) throws Exception {
        SecretKeySpec keyspec = new SecretKeySpec(keyValue, "AES");
        Cipher c = Cipher.getInstance(ALGORITHM);
        c.init(Cipher.DECRYPT_MODE, keyspec);
        byte[] decValue = c.doFinal(encryptedData);
        return decValue;
    }

    /**
     * Generate Secret Key
     * @return
     * @throws Exception
     */
    public static byte[] generateSecretKey(INTERNAL_KEY_GENERATION_MODE keyGenerationMode) throws Exception {
    	// ------ Solution 1 : Key randomly generated in bytes format 
    	if(keyGenerationMode == INTERNAL_KEY_GENERATION_MODE.RANDOM_BYTES_NOWRAPPED_KEY){
    		// Secret key on 16 characters : random string whose length is the number of characters specified.
        	//new byte[] { '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'a', 'b', 'c', 'd', 'e', 'f' };
        	byte[] keyValue = RandomStringUtils.randomAscii(16).getBytes(UNICODE_FORMAT);
        	return keyValue;
        	
        	// ------ Solution 2 : AES key generated using KeyGenerator Initialize the keysize to 128 bits (16 bytes)
    	}else if(keyGenerationMode == INTERNAL_KEY_GENERATION_MODE.GENERATED_NOWRAPPED_KEY) {
        	KeyGenerator keyGen = KeyGenerator.getInstance("AES");
    		keyGen.init(AES_KEYLENGTH);
    		SecretKey secretKey = keyGen.generateKey();
    		return secretKey.getEncoded();

	    	// ------ Solution 3 : Wrapped AES key : Generate an AES key using KeyGenerator Initialize the keysize to 128 bits (16 bytes) 
	    	// and wrapped in a AES key generated using KeyGenerator with a keysize of 256 bits
	    	// http://www.java2s.com/Tutorial/Java/0490__Security/AESKeygenerator.htm
	    	// http://flylib.com/books/en/1.274.1.29/1/ About the wrapped key
	    }else if(keyGenerationMode == INTERNAL_KEY_GENERATION_MODE.GENERATED_WRAPPED_KEY) {
        	Security.addProvider(new org.bouncycastle.jce.provider.BouncyCastleProvider());
        	
        	// Key to be wrapped
        	KeyGenerator generatorAES128ToBeWrapped = KeyGenerator.getInstance("AES", "BC"); // BC = BouncyCastle provider
            generatorAES128ToBeWrapped.init(AES_KEYLENGTH);
            Key keyToBeWrapped = generatorAES128ToBeWrapped.generateKey();
            System.out.println("input key to be wrapped : " + new String(keyToBeWrapped.getEncoded()));

        	// Key wrapper
            KeyGenerator generatorAES256Wrapper = KeyGenerator.getInstance("AES", "BC"); // BC = BouncyCastle provider
            generatorAES256Wrapper.init(256);
            Key wrapperKey = generatorAES256Wrapper.generateKey();
            System.out.println("wrapper key : " + new String(wrapperKey.getEncoded()));

            // Cipher
            Cipher cipher = Cipher.getInstance("AESWrap", "BC");
            cipher.init(Cipher.WRAP_MODE, wrapperKey);
            byte[] wrappedKey = cipher.wrap(keyToBeWrapped);
            System.out.println("wrapped key value : " + new String(wrappedKey));
            //
            cipher.init(Cipher.UNWRAP_MODE, wrapperKey);
            Key unWrappedKey = cipher.unwrap(wrappedKey, "AES", Cipher.SECRET_KEY);
            System.out.println("unwrapped key value : " + new String(unWrappedKey.getEncoded()));
            
            return unWrappedKey.getEncoded();
            
	    }else{
        	return new byte[] { '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'a', 'b', 'c', 'd', 'e', 'f' };
    	}
    }

}
