package com.ho.crypto.test3.encryption.rsa;

import java.math.BigInteger;
import java.security.KeyFactory;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.RSAPrivateKeySpec;
import java.security.spec.RSAPublicKeySpec;
import java.security.spec.X509EncodedKeySpec;

import javax.crypto.Cipher;


/*
 * Encryption with "RSA/ECB/PKCS1Padding" ALGO on 1024bits
 * https://docs.oracle.com/javase/7/docs/api/javax/crypto/Cipher.html
 * Every implementation of the Java platform is required to support the following standard Cipher transformations with the keysizes in parentheses:
		AES/CBC/NoPadding (128)
		AES/CBC/PKCS5Padding (128)
		AES/ECB/NoPadding (128)
		AES/ECB/PKCS5Padding (128)
		DES/CBC/NoPadding (56)
		DES/CBC/PKCS5Padding (56)
		DES/ECB/NoPadding (56)
		DES/ECB/PKCS5Padding (56)
		DESede/CBC/NoPadding (168)
		DESede/CBC/PKCS5Padding (168)
		DESede/ECB/NoPadding (168)
		DESede/ECB/PKCS5Padding (168)
		RSA/ECB/PKCS1Padding (1024, 2048)
		RSA/ECB/OAEPWithSHA-1AndMGF1Padding (1024, 2048)
		RSA/ECB/OAEPWithSHA-256AndMGF1Padding (1024, 2048)
 */
public class EncryptionRsaEcbPkcs1Padding {
    private static final String ALGORITHM = "RSA/ECB/PKCS1Padding";
    private static final int RSA_KEYLENGTH = 1024;
    
    /**
     * Constructor
     */
    public EncryptionRsaEcbPkcs1Padding(){
    }

    /**
	 * Encrypt with public key
     */
    public static byte[] encrypt(byte[] keyValue, byte[] data) throws Exception {
    	// Constrution of PublicKey from byte[]
        KeyFactory keyFactory = KeyFactory.getInstance("RSA");
    	PublicKey publicKey = keyFactory.generatePublic(new X509EncodedKeySpec(keyValue));
        return encrypt(publicKey, data);
    }
    public static byte[] encrypt(BigInteger modpublic, BigInteger exppublic, byte[] data) throws Exception {
    	// Constrution of PublicKey from modulus and Exponent
        RSAPublicKeySpec keySpec = new RSAPublicKeySpec(modpublic, exppublic);
        KeyFactory keyFactory = KeyFactory.getInstance("RSA");
        PublicKey publicKey = keyFactory.generatePublic(keySpec);
        return encrypt(publicKey, data);
    }
    public static byte[] encrypt(PublicKey publicKey, byte[] data) throws Exception {
    	Cipher c = Cipher.getInstance(ALGORITHM);
        c.init(Cipher.ENCRYPT_MODE, publicKey);
        byte[] encVal = c.doFinal(data);
        return encVal;
    }

    /**
	 * Decrypt with private key
     */
    public static byte[] decrypt(byte[] keyValue, byte[] encryptedData) throws Exception {
    	// Constrution of PrivateKey from byte[]
        KeyFactory keyFactory = KeyFactory.getInstance("RSA");
    	PrivateKey privateKey = keyFactory.generatePrivate(new PKCS8EncodedKeySpec(keyValue));
        return decrypt(privateKey, encryptedData);
    }
    public static byte[] decrypt(BigInteger modprivate, BigInteger expprivate, byte[] encryptedData) throws Exception {
    	// Constrution of PrivateKey from modulus and Exponent
        RSAPrivateKeySpec keySpec = new RSAPrivateKeySpec(modprivate, expprivate);
        KeyFactory keyFactory = KeyFactory.getInstance("RSA");
        PrivateKey privateKey = keyFactory.generatePrivate(keySpec);
        return decrypt(privateKey, encryptedData);
    }
    public static byte[] decrypt(PrivateKey privateKey, byte[] encryptedData) throws Exception {
        Cipher c = Cipher.getInstance(ALGORITHM);
        c.init(Cipher.DECRYPT_MODE, privateKey);
        byte[] decValue = c.doFinal(encryptedData);
        return decValue;
    }

    /**
     * Generate Secret Key
     * @return
     * @throws Exception
     */
    public static Pair<PublicKey,PrivateKey> generateKeyPair() throws Exception {
   		KeyPairGenerator keyGen = KeyPairGenerator.getInstance("RSA");
   		keyGen.initialize(RSA_KEYLENGTH);
   		KeyPair keyPair = keyGen.genKeyPair();
   		PublicKey publicKey = keyPair.getPublic();
   		PrivateKey privateKey = keyPair.getPrivate();
   		return new Pair<PublicKey,PrivateKey>(publicKey, privateKey);
    }

}
