package com.ho.crypto.test3.example.test;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.security.Key;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Date;

import javax.crypto.KeyGenerator;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.time.DateUtils;
import org.junit.Assert;

import com.ho.crypto.test3.example.CryptoTools;

public class CryptoToolsTest {

	private static PublicKey publicKey = null;
	private static PrivateKey privateKey = null;
	private static Key secretKey = null;
	static{
		try{
        	KeyGenerator keyGenAes = KeyGenerator.getInstance("AES");
    		keyGenAes.init(128); // 128 bits
    		secretKey = keyGenAes.generateKey();
    		//
	   		KeyPairGenerator keyGenRsa = KeyPairGenerator.getInstance("RSA");
	   		keyGenRsa.initialize(1024); // 1024 bits
	   		KeyPair keyPair = keyGenRsa.genKeyPair();
	   		publicKey = keyPair.getPublic();
	   		privateKey = keyPair.getPrivate();
		}catch(Throwable th){
			th.printStackTrace();
	    }
	}
	
    private static CryptoTools cryptoTools;

    public static void main(String[] args) {
    	CryptoToolsTest me = null;
    	try{
        	me = new CryptoToolsTest();
        	me.beforeTests();
        	
            String readableMessage = "Hello beautiful pink unicorn !!!";
        	me.test_01_obfuscateString_With_AesEcbSecret_Hexa(readableMessage);
        	me.test_02_obfuscateString_With_RSAPublic_Hexa(readableMessage);
        	me.test_03_obfuscateString_With_RSAPublic_SHA1_BASE64(readableMessage);
        	
        	try{
        		// Caused by: javax.crypto.IllegalBlockSizeException: Data must not be longer than 117 bytes
            	me.test_04_obfuscateStringOfXxxxXChars_With_RSAPublic_SHA1_BASE64(118);
            	Assert.assertTrue(false); // Normally, this point must not be UNreachable
        	}catch(javax.crypto.IllegalBlockSizeException excIgnore){
            	Assert.assertTrue(true); // Normally, this exception is OK
        		System.out.println("obfuscateStringOfXxxxXChars(118) -> IllegalBlockSizeException : OK");
        	}

        	me.test_04_obfuscateStringOfXxxxXChars_With_RSAPublic_SHA1_BASE64(117);
        	
        	me.test_05_cipherAndDecipherContent_With_AesCbc();
        	
        	me.test_06_cipherAndDecipherFile_With_AesCbc();
    	
    	}catch(Throwable th){
    		th.printStackTrace();
    	}finally{
    		if(me!=null){
        		me.afterTests();
    		}
    	}
	}
    
    public void beforeTests() throws Throwable {
        cryptoTools = new CryptoTools();
    }

    public void afterTests() {
    }

    public void test_01_obfuscateString_With_AesEcbSecret_Hexa(String readableMessage) throws Throwable{
    	System.out.println("----------------- test_01_obfuscateString_With_AesEcbSecret_Hexa(...)");
        final String obfuscatedValue = cryptoTools.obfuscate_With_AesEcbSecret_Hexa(secretKey, readableMessage);
        Assert.assertNotNull(obfuscatedValue);
        Assert.assertNotEquals("Obfuscation result should be different than original string", readableMessage, obfuscatedValue);

        // Return the same obfuscated result for the same source
        final String newObfuscatedValue = cryptoTools.obfuscate_With_AesEcbSecret_Hexa(secretKey, readableMessage);
        Assert.assertEquals("Obfuscation of a same value should return the same result", obfuscatedValue, newObfuscatedValue);

        // Return the same illuminated result for the same obfuscated value
        final String illuminatedValue = cryptoTools.illuminate_With_Hexa_AesEcbSecret(secretKey, obfuscatedValue);
        Assert.assertEquals("Illuminated value should return the original value", readableMessage, illuminatedValue);
        final String newIlluminatedValue = cryptoTools.illuminate_With_Hexa_AesEcbSecret(secretKey, newObfuscatedValue);
        Assert.assertEquals("Illuminated values of obfuscated values should return the same result (source)", illuminatedValue, newIlluminatedValue);
    }
    
    public void test_02_obfuscateString_With_RSAPublic_Hexa(String readableMessage) throws Throwable{
    	System.out.println("----------------- test_02_obfuscateString_With_RSAPublic_Hexa(...)");
        final String obfuscatedValue = cryptoTools.obfuscate_With_RSAPublic_Hexa(publicKey, readableMessage);
        Assert.assertNotNull(obfuscatedValue);
        Assert.assertNotEquals("Obfuscation result should be different than original string", readableMessage, obfuscatedValue);

        // Don't return the same obfuscated result for the same source
        final String newObfuscatedValue = cryptoTools.obfuscate_With_RSAPublic_Hexa(publicKey, readableMessage);
        Assert.assertNotEquals("Obfuscation of a same value should return different result", obfuscatedValue, newObfuscatedValue);

        // Return the same illuminated result for different obfuscated value of the same original source
        final String illuminatedValue = cryptoTools.illuminate_With_Hexa_RSAPrivate(privateKey, obfuscatedValue);
        Assert.assertEquals("Illuminated value should return the original value", readableMessage, illuminatedValue);
        final String newIlluminatedValue = cryptoTools.illuminate_With_Hexa_RSAPrivate(privateKey, newObfuscatedValue);
        Assert.assertEquals("Illuminated values of different obfuscated values of a same source value should return the same result (source)", illuminatedValue, newIlluminatedValue);
    }
    
    
    public void test_03_obfuscateString_With_RSAPublic_SHA1_BASE64(String readableMessage) throws Throwable{
    	System.out.println("----------------- test_03_obfuscateString_With_RSAPublic_SHA1_BASE64(...)");
        final String obfuscatedValue = cryptoTools.obfuscate_With_RSAPublic_SHA1_BASE64(publicKey, readableMessage);
        Assert.assertNotNull(obfuscatedValue);
        Assert.assertNotEquals("Obfuscation result should be different than original string", readableMessage, obfuscatedValue);

        // Don't return the same obfuscated result for the same source
        final String newObfuscatedValue = cryptoTools.obfuscate_With_RSAPublic_SHA1_BASE64(publicKey, readableMessage);
        Assert.assertNotEquals("Obfuscation of a same value shouldn't return the same result", obfuscatedValue, newObfuscatedValue);
    }

    public void test_04_obfuscateStringOfXxxxXChars_With_RSAPublic_SHA1_BASE64(int nbChars) throws Throwable {
    	// RSA Original Max length to obfuscate
    	// 1024 bits = RSA keys length
    	int RSA_ORIGINAL_MAX_LENGTH_TO_OBFUSCATE = 117; //1024/8 - 11;

    	System.out.println("----------------- test_04_obfuscateStringOfXxxxXChars_With_RSAPublic_SHA1_BASE64("+nbChars+")");
        final StringBuilder sb = new StringBuilder();
        for (int i = 0; i < nbChars; i++) {
            sb.append("#");
        }
        final String obfuscatedValue = cryptoTools.obfuscate_With_RSAPublic_SHA1_BASE64(publicKey, sb.toString());
        Assert.assertNotNull(obfuscatedValue);
        Assert.assertTrue("Obfuscation result should have max " + RSA_ORIGINAL_MAX_LENGTH_TO_OBFUSCATE + " chars", obfuscatedValue.length() < RSA_ORIGINAL_MAX_LENGTH_TO_OBFUSCATE);
    }
    
    public void test_05_cipherAndDecipherContent_With_AesCbc() throws Throwable {
    	System.out.println("----------------- test_05_cipherAndDecipherContent_With_AesCbc(...)");
        String TEST_FILE = System.getProperty("user.dir") + "/resources/default-file.csv";
        final byte[] originalContent = readFully(new File(TEST_FILE));
        final byte[] cipheredContent = cryptoTools.encrypt_With_AesCbcSecret(secretKey, getResourceStream(TEST_FILE));
        Assert.assertFalse(Arrays.equals(originalContent, cipheredContent));
        final byte[] decipheredContent = cryptoTools.decryptStream_With_AesCbcSecret(secretKey, new ByteArrayInputStream(cipheredContent));
        Assert.assertArrayEquals(originalContent, decipheredContent);
    }

    public void test_06_cipherAndDecipherFile_With_AesCbc() throws Throwable {
    	System.out.println("----------------- test_06_cipherAndDecipherFile_With_AesCbc(...)");
        final File cipheredFile = getTempFile();
        final File decipheredFile = getTempFile();
        String TEST_FILE = System.getProperty("user.dir") + "/resources/default-file.csv";

        try {
            final File readableFile = new File(TEST_FILE);
            cryptoTools.encrypt_With_AesCbcSecret(secretKey, readableFile, cipheredFile);
            Assert.assertFalse("Ciphered file must not be the same as the readable file", equals(readableFile, cipheredFile));

            cryptoTools.decryptFile_With_AesCbcSecret(secretKey, cipheredFile, decipheredFile);
            Assert.assertTrue("Deciphered file must be the same as the readable file", equals(readableFile, decipheredFile));
        } finally {
            remove(cipheredFile);
            remove(decipheredFile);
        }
    }
    
    //--------------------------------------------------------------------------------- private methods
    private InputStream getResourceStream(final String fileName) throws Throwable{
		final InputStream is = new FileInputStream(fileName);
		Assert.assertNotNull("Cannot load '" + fileName + "'", is);
		return is;
	}

	private File getTempFile() throws Throwable {
		try {
			final File tempFile = File.createTempFile("file-for-testing", null);
			Assert.assertNotNull(tempFile);
			tempFile.deleteOnExit();
			return tempFile;
		} catch (final IOException ioe) {
			throw new RuntimeException(ioe);
		}
	}
	
	private void remove(final File file) {
		if (file != null && file.exists()) {
			file.delete();
		}
	}
	
	private byte[] readFully(final File file) throws Throwable {
		try {
            return readFully(new FileInputStream(file));
		} catch (Exception e) {
			throw new RuntimeException("Cannot read file '" + file.getAbsolutePath() + "'", e);
		}
	}

	private byte[] readFully(final InputStream is) throws Throwable {
        try {
            final ByteArrayOutputStream baos = new ByteArrayOutputStream();
            IOUtils.copy(is, baos);
            return baos.toByteArray();
        } catch (Exception e) {
            throw new RuntimeException("Cannot read stream", e);
        } finally {
            IOUtils.closeQuietly(is);
        }
    }

    private boolean equals(final File file1, final File file2) throws Throwable {
        final byte[] content1 = readFully(new FileInputStream(file1));
        final byte[] content2 = readFully(new FileInputStream(file2));
        return Arrays.equals(content1, content2);
    }

    private Date today() {
		return new Date();

	}

	private Date yesterday() {
		return DateUtils.truncate(DateUtils.addDays(new Date(), -1), Calendar.DATE);
	}
    
}