package com.ho.crypto.test3.example;


import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.security.Key;
import java.security.MessageDigest;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.SecureRandom;

import javax.crypto.Cipher;
import javax.crypto.CipherInputStream;
import javax.crypto.CipherOutputStream;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.SecretKeySpec;
import javax.xml.bind.DatatypeConverter;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;

import com.ho.crypto.test3.encryption.base64.Base64Url;

/**
 * Cryptography tools using a secret, private and public keys
 */
public class CryptoTools {

	private final static String RSA_ALGO = "RSA/ECB/PKCS1Padding";
	private final static String AES_ECB_ALGO = "AES/ECB/PKCS5Padding";
	private final static String DIGEST_ALGO = "SHA-1";
	//
	private final static String AES_CBC_ALGO = "AES/CBC/PKCS5Padding";
	private static final int IV_LENGTH = 16;

	/**
	 * Obfuscate a String via the following steps:
	 * 1. the AES/ECB/PKCS5Padding encryption via symetric secret key provided in parameter,
	 * 2. the HEXA conversion allowing to obtain a result with strandard ANSI charachters (0-9,A-F)
	 * 
	 * => Return the same obfuscated result for the same source
	 * 
	 * @param secretKey
	 * @param value
	 * @return
	 * @throws Throwable
	 */
	public String obfuscate_With_AesEcbSecret_Hexa(Key secretKey, final String value) throws Throwable{
        if (StringUtils.isEmpty(value)) {
            return value;
        }
        System.out.println(String.format("----- FUNCTION : obfuscate_With_AesEcbSecret_Hexa - START"));

        // STEP 1 : AES
        System.out.println(String.format("- STEP 1 : Encrypt '%s' with '%s' symmetric algo (secretKey)", value, AES_ECB_ALGO));
        final Cipher aesCipher = Cipher.getInstance(AES_ECB_ALGO);
        aesCipher.init(Cipher.ENCRYPT_MODE, secretKey);
        final byte[] aesBytes = aesCipher.doFinal(value.getBytes());
        System.out.println(String.format("- STEP 1 : Done => '%s'", new String(aesBytes)));

        // STEP 2 : HEXA
        System.out.println(String.format("- STEP 2 : Convert to hexa representation"));
        final String hexaValue = toHexString(aesBytes);
        System.out.println(String.format("- STEP 2 : Done => '%s'", hexaValue));
        
        System.out.println(String.format("- RESULT = '%s' obfuscated to '%s'", value, hexaValue));

        System.out.println(String.format("----- FUNCTION : obfuscate_With_AesEcbSecret_Hexa - END"));
        return hexaValue;
    }

	/**
	 * Obfuscate a String via the following steps:
	 * 1. the RSA/ECB/PKCS1Padding encryption via asymetric public key provided in parameter,
	 * 2. the SHA-1 hash generation in order to forbid the reversibility (max length is 27 bytes)
	 * 3. the BASE64 transformation allowing to obtain a result with strandard charachters
	 * 
	 * => Doesn't return the same obfuscated result for the same source
	 * 
	 * @param publicKey
	 * @param value
	 * @return
	 * @throws Throwable
	 */
    public String obfuscate_With_RSAPublic_SHA1_BASE64(PublicKey publicKey, final String value) throws Throwable{
        if (StringUtils.isEmpty(value)) {
            return value;
        }

        System.out.println(String.format("----- FUNCTION : obfuscate_With_RSAPublic_SHA1_BASE64 - START"));
        // STEP 1 : RSA
        System.out.println(String.format("- STEP 1 : Encrypt with '%s' asymmetric algo (publicKey)", RSA_ALGO));
        final Cipher rsaCipher = Cipher.getInstance(RSA_ALGO);
        rsaCipher.init(Cipher.ENCRYPT_MODE, publicKey);
        final byte[] cipheredBytes = rsaCipher.doFinal(value.getBytes());
        System.out.println(String.format("- STEP 1 : Done => '%s'", new String(cipheredBytes)));

        // STEP 2 : SHA-1
        System.out.println(String.format("- STEP 2 : Generate the obfuscated value with a hash '%s'", DIGEST_ALGO));
    	final MessageDigest md = MessageDigest.getInstance(DIGEST_ALGO);
    	final byte[] digest = md.digest(cipheredBytes);
        System.out.println(String.format("- STEP 2 : Done => '%s'", new String(digest)));

    	// STEP 3 : BASE64
        System.out.println(String.format("- STEP 3 : Transform to BASE64 from the value '%s'", new String(digest)));
    	final String obfuscatedValue = Base64Url.encode(digest);
        System.out.println(String.format("- STEP 3 : Done => '%s'", obfuscatedValue));

        System.out.println(String.format("- RESULT = '%s' obfuscated to '%s'", value, obfuscatedValue));

        System.out.println(String.format("----- FUNCTION : obfuscate_With_RSAPublic_SHA1_BASE64 - END"));

        return obfuscatedValue;
    }


	/**
	 * Obfuscate a String via the following steps:
	 * 1. the RSA/ECB/PKCS1Padding encryption via asymetric public key provided in parameter,
	 * 2. the HEXA conversion allowing to obtain a result with strandard ANSI charachters (0-9,A-F)
	 * 
	 * => Doesn't return the same obfuscated result for the same source
	 * 
	 * @param publicKey
	 * @param value
	 * @return
	 * @throws Throwable
	 */
    public String obfuscate_With_RSAPublic_Hexa(PublicKey publicKey, final String value) throws Throwable{
        if (StringUtils.isEmpty(value)) {
            return value;
        }

        System.out.println(String.format("----- FUNCTION : obfuscate_With_RSAPublic_Hexa - START"));

        // STEP 1 : RSA
        System.out.println(String.format("- STEP 1 : Encrypt with '%s' asymmetric algo (publicKey)", RSA_ALGO));
        final Cipher rsaCipher = Cipher.getInstance(RSA_ALGO);
        rsaCipher.init(Cipher.ENCRYPT_MODE, publicKey);
        final byte[] cipheredBytes = rsaCipher.doFinal(value.getBytes());
        System.out.println(String.format("- STEP 1 : Done => '%s'", new String(cipheredBytes)));

        // STEP 2 : HEXA
        System.out.println(String.format("- STEP 2 : Convert to hexa representation"));
    	final String hexaValue = toHexString(cipheredBytes);
        System.out.println(String.format("- STEP 2 : Done => '%s'", hexaValue));

        System.out.println(String.format("- RESULT = '%s' obfuscated to '%s'", value, hexaValue));

        System.out.println(String.format("----- FUNCTION : obfuscate_With_RSAPublic_Hexa - END"));

    	return hexaValue;
    }
    
	/**
	 * Illuminate an obfuscated String via the following steps:
	 * 1. the HEXA reverse-conversion,
	 * 2. the RSA/ECB/PKCS1Padding decryption via asymetric private key provided in parameter,
	 * 
	 * => Return the same illuminated result for different obfuscated value of the same original source
	 * 
     * @param privateKey
     * @param value
     * @return
     * @throws Throwable
     */
    public String illuminate_With_Hexa_RSAPrivate(PrivateKey privateKey, final String value) throws Throwable {
        if (StringUtils.isEmpty(value)) {
            return value;
        }

        System.out.println(String.format("----- FUNCTION : illuminate_With_Hexa_RSAPrivate - START"));

        // STEP 1 : HEXA
        System.out.println(String.format("- STEP 1 : Convert from hexa representation"));
    	final byte[] hexaValueBytes = fromHexString(value);
        System.out.println(String.format("- STEP 1 : Done => '%s'", new String(hexaValueBytes)));

        // STEP 2 : RSA
        System.out.println(String.format("- STEP 2 : Encrypt with '%s' asymmetric algo (privateKey)", RSA_ALGO));
        final Cipher aesCipher = Cipher.getInstance(RSA_ALGO);
        aesCipher.init(Cipher.DECRYPT_MODE, privateKey);
        final byte[] content = aesCipher.doFinal(hexaValueBytes);
        final String valueRet = new String(content);
        System.out.println(String.format("- STEP 2 : Done => '%s'", valueRet));

        System.out.println(String.format("----- FUNCTION : illuminate_With_Hexa_RSAPrivate - END"));

        return valueRet;
    }

    /**
	 * Illuminate an obfuscated String via the following steps:
	 * 1. the HEXA reverse-conversion,
	 * 2. the AES/ECB/PKCS5Padding decryption via symetric secret key provided in parameter,
     * 
     * => Return the same illuminated result for the same obfuscated value
     * 
     * @param secretKey
     * @param value
     * @return
     * @throws Throwable
     */
    public String illuminate_With_Hexa_AesEcbSecret(Key secretKey, final String value) throws Throwable {
        if (StringUtils.isEmpty(value)) {
            return value;
        }

        System.out.println(String.format("----- FUNCTION : illuminate_With_Hexa_AesEcbSecret - START"));
        
        // STEP 1 : HEXA
        System.out.println(String.format("- STEP 1 : Convert from hexa representation"));
    	final byte[] hexaValueBytes = fromHexString(value);
        System.out.println(String.format("- STEP 1 : Done => '%s'", new String(hexaValueBytes)));

        // STEP 2 : AES
        System.out.println(String.format("- STEP 2 : Encrypt with '%s' symmetric algo (secretKey)", RSA_ALGO));
        final Cipher aesCipher = Cipher.getInstance(AES_ECB_ALGO);
        aesCipher.init(Cipher.DECRYPT_MODE, secretKey);
        final byte[] content = aesCipher.doFinal(fromHexString(value));
        final String valueRet = new String(content);
        System.out.println(String.format("- STEP 2 : Done => '%s'", valueRet));

        System.out.println(String.format("----- FUNCTION : illuminate_With_Hexa_AesEcbSecret - END"));

        return valueRet;
    }

    
    /**
	 * Encrypt a String via the following steps:
	 * 1. the creation of a random initialization vector needed for the encryption
	 * 2. the AES/CBC/PKCS5Padding encryption via symetric secret key provided in parameter,
	 * 
     * @param secretKey
     * @param is
     * @return
     */
    public byte[] encrypt_With_AesCbcSecret(Key secretKey, final InputStream is) {
        final ByteArrayOutputStream baos = new ByteArrayOutputStream();
        encrypt_With_AesCbcSecret(secretKey, is, baos);
        return baos.toByteArray();
    }
    public void encrypt_With_AesCbcSecret(Key secretKey, final File readableFile, final File cipheredFile) throws Throwable {
        InputStream is = null;
        OutputStream os = null;
        try {
            is = new FileInputStream(readableFile);
            os = new FileOutputStream(cipheredFile);
            encrypt_With_AesCbcSecret(secretKey, is, os);
        } catch (FileNotFoundException fnfe) {
            throw new RuntimeException("Cannot cipher file", fnfe);
        } finally {
            IOUtils.closeQuietly(os);
            IOUtils.closeQuietly(is);
        }
    }
    public void encrypt_With_AesCbcSecret(Key secretKey, final InputStream is, final OutputStream os) {
        CipherOutputStream cos = null;
        try {
            System.out.println(String.format("----- FUNCTION : encrypt_With_AesCbcSecret - START"));

            // STEP 1 : IV
            System.out.println(String.format("- STEP 1 : Create a random initialization vector with a length "+IV_LENGTH+ " and write it in the outputstream"));
        	final byte[] ivBytes = new SecureRandom().generateSeed(IV_LENGTH);
            os.write(ivBytes);
            System.out.println(String.format("- STEP 1 : Done => '%s'", new String(ivBytes)));

            // STEP 2 : AES
            System.out.println(String.format("- STEP 2 : Encrypt with '%s' symmetric algo (secretKey)", AES_CBC_ALGO));
            final Cipher cipher = Cipher.getInstance(AES_CBC_ALGO);
            cipher.init(Cipher.ENCRYPT_MODE, getSecretKeySpec(secretKey), new IvParameterSpec(ivBytes));
            cos = new CipherOutputStream(os, cipher);
            IOUtils.copy(is, cos);
            System.out.println(String.format("- STEP 2 : Done "));

            System.out.println(String.format("----- FUNCTION : encrypt_With_AesCbcSecret - END"));

        } catch (Exception e) {
            throw new RuntimeException("Cannot cipher stream", e);
        } finally {
            IOUtils.closeQuietly(cos);
            IOUtils.closeQuietly(is);
        }
    }


    /**
	 * Decrypt a String via the following steps:
	 * 1. the reading of the random initialization vector needed for the decryption
	 * 2. the AES/CBC/PKCS5Padding decryption via symetric secret key provided in parameter,
	 * 
     * @param secretKey
     * @param is
     * @return
     */
    public byte[] decryptFile_With_AesCbcSecret(Key secretKey, final File file) {
        if (file == null) {
            throw new IllegalArgumentException("File cannot be null");
        }
        InputStream is = null;
        try {
            is = new FileInputStream(file);
            return decryptStream_With_AesCbcSecret(secretKey, is);
        } catch (FileNotFoundException e) {
            throw new RuntimeException(String.format("Cannot decipher file '%s'", file.getAbsolutePath()), e);
        } finally {
            IOUtils.closeQuietly(is);
        }
    }
    public void decryptFile_With_AesCbcSecret(Key secretKey, final File cipheredFile, final File readableFile) throws Throwable {
        InputStream is = null;
        OutputStream os = null;
        try {
            is = new FileInputStream(cipheredFile);
            os = new FileOutputStream(readableFile);
            decryptStream_With_AesCbcSecret(secretKey, is, os);
        } catch (FileNotFoundException fnfe) {
            throw new RuntimeException("Cannot decipher file", fnfe);
        } finally {
            IOUtils.closeQuietly(os);
            IOUtils.closeQuietly(is);
        }
    }
    public byte[] decryptStream_With_AesCbcSecret(Key secretKey, final InputStream is) {
        CipherInputStream cis = null;
        try {
            final ByteArrayOutputStream baos = new ByteArrayOutputStream();
            decryptStream_With_AesCbcSecret(secretKey, is, baos);
            return baos.toByteArray();
        } catch (Exception e) {
            throw new RuntimeException("Cannot decipher stream", e);
        } finally {
            IOUtils.closeQuietly(cis);
        }
    }
    public void decryptStream_With_AesCbcSecret(Key secretKey, final InputStream is, final OutputStream os) {
        CipherInputStream cis = null;
        try {
            System.out.println(String.format("----- FUNCTION : decryptStream_With_AesCbcSecret - START"));

            // STEP 1 : IV
            System.out.println(String.format("- STEP 1 : Read the initialization vector with a length "+IV_LENGTH+ ""));
           	// The first x bytes of the file composed the initialization vector
            final byte[] ivBytes = new byte[IV_LENGTH];
            is.read(ivBytes);
            System.out.println(String.format("- STEP 1 : Done => '%s'", new String(ivBytes)));


            // STEP 2 : AES
            System.out.println(String.format("- STEP 2 : Decrypt with '%s' symmetric algo (secretKey)", AES_CBC_ALGO));
            final Cipher cipher = Cipher.getInstance(AES_CBC_ALGO);
            cipher.init(Cipher.DECRYPT_MODE, getSecretKeySpec(secretKey), new IvParameterSpec(ivBytes));
            cis = new CipherInputStream(is, cipher);
            IOUtils.copy(cis, os);
            System.out.println(String.format("- STEP 2 : Done "));

            System.out.println(String.format("----- FUNCTION : decryptStream_With_AesCbcSecret - END"));
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            IOUtils.closeQuietly(cis);
            IOUtils.closeQuietly(os);
        }
    }


    private SecretKeySpec getSecretKeySpec(Key secretKey) {
    	if(secretKey!=null){
            final byte[] encodedKey = secretKey.getEncoded();
            return new SecretKeySpec(encodedKey, "AES");
    	}
    	return null;
    }

    private String toHexString(final byte[] bytes) {
        return DatatypeConverter.printHexBinary(bytes);
    }

    private byte[] fromHexString(final String hexString) {
        return DatatypeConverter.parseHexBinary(hexString);
    }
}
