package com.ho.crypto.test3.example;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.LinkedHashMap;
import java.util.Map;

import org.apache.commons.csv.CSVFormat;
import org.apache.commons.csv.CSVParser;
import org.apache.commons.csv.CSVRecord;
import org.apache.commons.lang.StringUtils;

/**
 * Class managing the obfuscation file containing the following structure:
 * keyValue1,obfuscatedValue1-1,obfuscatedValue1-2
 * keyValue2,obfuscatedValue2-1,obfuscatedValue2-2
 * keyValue3,obfuscatedValue3-1,obfuscatedValue3-2
 * keyValue4,obfuscatedValue4-1,obfuscatedValue4-2
 * ...
 * 
 * @author Huseyin
 *
 */
public class GenericObfuscationFile {

    public static final String LINE_SEPARATOR = System.getProperty("line.separator");
    public static final String FIELD_SEPARATOR = ""+CSVFormat.RFC4180.getDelimiter();
    private static final int DEFAULT_KEY_VALUE_INDEX = 0;

    private final LinkedHashMap<Integer, LinkedHashMap<String, String>> obfuscatedFileContents = new LinkedHashMap<Integer, LinkedHashMap<String, String>>();
    private final File file;
    private int recentlyAddedValues = 0;
    private int nbOfColumns = 0;
    private int keyValueColumnIndex = DEFAULT_KEY_VALUE_INDEX;
    
    public GenericObfuscationFile(final String obfuscationFilePath, final int nbOfColumns, final int keyValueColumnIndex) throws Throwable {
        this(new File(obfuscationFilePath), nbOfColumns, keyValueColumnIndex);
    }

    public GenericObfuscationFile(final File file, final int nbOfColumns, final int keyValueColumnIndex) throws Throwable {
        super();
        this.keyValueColumnIndex = keyValueColumnIndex;
        
        if (file == null) {
            throw new IllegalArgumentException("A file should be provided");
        }
        this.file = file;

        if (file.exists()) {
        	System.out.println(String.format("Loading existing obfuscated values from '%s'", file.getAbsolutePath()));
        	try {
        		final CSVParser parser = CSVParser.parse(file, java.nio.charset.Charset.forName("utf-8"), CSVFormat.RFC4180);
        		this.nbOfColumns = nbOfColumns;
        		if(nbOfColumns<=1){
        			throw new RuntimeException("File must contain at least 2 columns.");
        		}
    			for (CSVRecord csvRecord : parser) {
    				for (int i = 0; i <nbOfColumns; i++) {
        				LinkedHashMap<String, String> valuesOfColumn = obfuscatedFileContents.get(new Integer(i));
        				if(valuesOfColumn == null)
        					valuesOfColumn = new LinkedHashMap<String, String>();
        				valuesOfColumn.put(csvRecord.get(this.keyValueColumnIndex), csvRecord.get(i));
        				obfuscatedFileContents.put(new Integer(i), valuesOfColumn);
        			}
        		}
        	} catch (Throwable th) {
        		throw new RuntimeException("Cannot find provided file", th);
        	}
        } else {
            System.out.println(String.format("Use new obfuscated values file '%s'", file.getAbsolutePath()));
        }
    }

    public boolean hasChanges() {
        return recentlyAddedValues > 0;
    }

    public boolean containsKeyValue(final String keyValue) {
        return obfuscatedFileContents!=null && obfuscatedFileContents.get(this.keyValueColumnIndex)!=null ? 
        		obfuscatedFileContents.get(this.keyValueColumnIndex).containsKey(keyValue) : false;
    }

    public boolean containsObfuscatedValue(final String obfuscatedValue) {
    	if(obfuscatedFileContents!=null){
    		for (int i = 0; i < this.nbOfColumns; i++) {
    			LinkedHashMap<String, String> valuesOfColumn = obfuscatedFileContents.get(new Integer(i));
    			if(valuesOfColumn!=null && valuesOfColumn.containsValue(obfuscatedValue)){
    				return true;
    			}
    		}
    	}
        return false;
    }

    public String getObfuscatedValue(final int indexOfColumn, final String keyValue) {
        return obfuscatedFileContents!=null && obfuscatedFileContents.get(indexOfColumn)!=null ? 
        		obfuscatedFileContents.get(indexOfColumn).get(keyValue) : null;
    }

    public String getKeyValue(final int indexOfColumn, final String obfuscatedValue) {
    	LinkedHashMap<String, String> valuesOfColumn = obfuscatedFileContents!=null ? obfuscatedFileContents.get(indexOfColumn):null;
        if (valuesOfColumn==null || valuesOfColumn.containsValue(obfuscatedValue)==false) {
            return null;
        }

        for (Map.Entry<String, String> entry : valuesOfColumn.entrySet()) {
            if (obfuscatedValue.equals(entry.getValue())) {
                return entry.getKey();
            }
        }

        return null;
    }

    public synchronized void add(final String keyValue, String... obfuscatedValues) throws Throwable {
    	if (obfuscatedValues==null || this.nbOfColumns!=obfuscatedValues.length+1) {
            throw new IllegalArgumentException(String.format("The number of obfuscated values (%s) are not compliance with the obfuscation file's structure (%s)",obfuscatedValues.length+1, this.nbOfColumns));
        }
    	
    	if (containsKeyValue(keyValue)) {
            throw new IllegalArgumentException(String.format("Obfuscation file already contains an obfuscated value for this key value (%s)", keyValue));
        }
        
        // For each obfuscated value
        for (int i = 0; i < obfuscatedValues.length; i++) {
        	String obfuscatedValue = obfuscatedValues[i];
            if (containsObfuscatedValue(obfuscatedValue)) {
                throw new IllegalArgumentException(String.format("Obfuscation file already contains an obfuscated value (%s) for another key value", obfuscatedValue));
            }
		}
        
        try {
        	String[] values = new String[obfuscatedValues.length+1];
        	int indexCount = 0;
        	for (int i = 0; i < this.nbOfColumns; i++) {
        		if(i == this.keyValueColumnIndex){
                	values[i] = keyValue;
        		}else{
                	values[i] = obfuscatedValues[indexCount];
                	indexCount++;
        		}
			}
        	write(values);
            //
        	if(obfuscatedFileContents!=null){
            	indexCount = 0;
        		for (int i = 0; i < this.nbOfColumns; i++) {
        			LinkedHashMap<String, String> valuesOfColumn = obfuscatedFileContents.get(new Integer(i));
        			if(valuesOfColumn==null){
        				valuesOfColumn = new LinkedHashMap<String, String>();
        			}

        			if(i == this.keyValueColumnIndex){
        				valuesOfColumn.put(keyValue, keyValue);
            		}else{
        				valuesOfColumn.put(keyValue, obfuscatedValues[indexCount]);
                    	indexCount++;
            		}
        			
        			obfuscatedFileContents.put(new Integer(i), valuesOfColumn);
        		}
        	}
            //
            recentlyAddedValues++;
        } catch (IOException ioe) {
            throw new RuntimeException("Cannot add entry to obfuscation file", ioe);
        }
    }
    
    
    public File exportToOtherFile(final File targetFile) throws Throwable {
        PrintWriter pw = null;
        try {
        	if(obfuscatedFileContents==null){
                throw new IllegalArgumentException(String.format("No data found for the export"));
        	}

        	System.out.println(String.format("Export all %d lines, %d columns to '%s' (%d newly added values)", 
        			(obfuscatedFileContents!=null&&obfuscatedFileContents.get(0)!=null ? obfuscatedFileContents.get(0).size() : 0), 
        			obfuscatedFileContents.size(), 
        			targetFile.getAbsolutePath(), 
        			recentlyAddedValues));

            pw = new PrintWriter(targetFile);

            LinkedHashMap<String, String> keyValues = obfuscatedFileContents.get(new Integer(this.keyValueColumnIndex));
            for (String keyValue : keyValues.keySet()) {
            	StringBuffer tmpToWrite = null;
        		for (int i = 0; i < this.nbOfColumns; i++) {
        			if(tmpToWrite==null){
        				tmpToWrite = new StringBuffer("");
        			}else{
        				tmpToWrite.append(GenericObfuscationFile.FIELD_SEPARATOR);
        			}

        			if(i!=this.keyValueColumnIndex){
    	    			LinkedHashMap<String, String> valuesOfColumn = obfuscatedFileContents.get(new Integer(i));
    	    			if(valuesOfColumn!=null && valuesOfColumn.get(keyValue)!=null){
    	    				tmpToWrite.append(valuesOfColumn.get(keyValue));
    	    			}else{
    	    				tmpToWrite.append("");
    	    			}
        			}else{
	    				tmpToWrite.append(keyValue);
        			}
	    		}
        		if(tmpToWrite!=null){
                    pw.write(tmpToWrite.toString() + LINE_SEPARATOR);
        		}
            }

            return targetFile;
        } catch (IOException ioe) {
            throw new RuntimeException("Cannot export to other file", ioe);
        } finally {
            if (pw != null) {
                pw.flush();
                pw.close();
            }
        }
    }
    
    
    private void write(final String[] values) throws IOException {
        FileWriter fw = null;
        PrintWriter pw = null;
        try {
            System.out.println(String.format("Add new line to '%s'", file.getAbsolutePath()));

            fw = new FileWriter(file, true);
            pw = new PrintWriter(fw);
            pw.write(StringUtils.join(values, FIELD_SEPARATOR) + LINE_SEPARATOR);
        } finally {
            if (pw != null) {
                pw.flush();
                fw.flush();
                pw.close();
            }
            if (fw != null) {
                fw.close();
            }
        }
    }

}
