package com.ho.crypto.test3.example.test;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.Date;
import java.util.Scanner;

import org.junit.Assert;

import com.ho.crypto.test3.example.GenericObfuscationFile;

public class GenericObfuscationFileTest {

    private static final String FIRST_KEY_VALUE = "KEY-1";
    private static final String FIRST_OBFUSCATED_VALUE_IN_COL1 = "OBFUSCATED-1-1";
    private static final String FIRST_OBFUSCATED_VALUE_IN_COL2 = "OBFUSCATED-1-2";
    private static final String SECOND_KEY_VALUE = "KEY-2";
    private static final String SECOND_OBFUSCATED_VALUE_IN_COL1 = "OBFUSCATED-2-1";
    private static final String SECOND_OBFUSCATED_VALUE_IN_COL2 = "OBFUSCATED-2-2";
    //
    private static final int NB_OF_COLUMNS = 3;
    private static final int INDEX_OF_COLUMN_KEY = 0;
    private static final int INDEX_OF_COLUMN_OBFUSCATED_VALUE_1 = 1;
    private static final int INDEX_OF_COLUMN_OBFUSCATED_VALUE_2 = 2;
    
    private static File temporaryFile = null;
    private static GenericObfuscationFile obfuscationFile = null;
    
    public static void main(String[] args) {
    	GenericObfuscationFileTest me = null;
    	try{
        	me = new GenericObfuscationFileTest();
        	me.beforeTests();
        	me.test_01_createObfuscationFileFromEmptyFile();
        	me.test_02_addFirstObfuscationEntry();
            try{
                me.test_03_addDuplicatedObfuscatedInColumn2Value();
            }catch(IllegalArgumentException expected){
            	System.out.println("IllegalArgumentException expected : " + expected.getMessage());
            }
            try{
                me.test_04_addDuplicatedObfuscatedValue();
            }catch(IllegalArgumentException expected){
            	System.out.println("IllegalArgumentException expected : " + expected.getMessage());
            }
        	me.test_05_addSecondObfuscationEntry();
        	me.test_06_createObfuscationFileNonEmptyFromFile();
        	me.test_07_exportToOtherFile();
    	}catch(Throwable th){
    		th.printStackTrace();
    	}finally{
    		if(me!=null){
        		me.afterTests();
    		}
    	}
	}
    
    public void beforeTests() throws Throwable {
        temporaryFile = getTempFile();
    }

    public void afterTests() {
        remove(temporaryFile);
    }

    public void test_01_createObfuscationFileFromEmptyFile() throws Throwable {
    	System.out.println("----------------- test_01_createObfuscationFileFromEmptyFile(...)");
        obfuscationFile = new GenericObfuscationFile(temporaryFile, NB_OF_COLUMNS, INDEX_OF_COLUMN_KEY);
        Assert.assertNotNull(obfuscationFile);
        Assert.assertFalse(obfuscationFile.hasChanges());
    }

    public void test_02_addFirstObfuscationEntry() throws Throwable {
    	System.out.println("----------------- test_02_addFirstObfuscationEntry(...)");
        // Add a first entry
        Assert.assertFalse(obfuscationFile.containsKeyValue(FIRST_KEY_VALUE));
        obfuscationFile.add(FIRST_KEY_VALUE, FIRST_OBFUSCATED_VALUE_IN_COL1, FIRST_OBFUSCATED_VALUE_IN_COL2);
        Assert.assertTrue(obfuscationFile.containsKeyValue(FIRST_KEY_VALUE));
        Assert.assertEquals(FIRST_OBFUSCATED_VALUE_IN_COL1, obfuscationFile.getObfuscatedValue(INDEX_OF_COLUMN_OBFUSCATED_VALUE_1, FIRST_KEY_VALUE));
        Assert.assertEquals(FIRST_OBFUSCATED_VALUE_IN_COL2, obfuscationFile.getObfuscatedValue(INDEX_OF_COLUMN_OBFUSCATED_VALUE_2, FIRST_KEY_VALUE));
        Assert.assertEquals(FIRST_KEY_VALUE, obfuscationFile.getKeyValue(INDEX_OF_COLUMN_OBFUSCATED_VALUE_1, FIRST_OBFUSCATED_VALUE_IN_COL1));
        Assert.assertEquals(FIRST_KEY_VALUE, obfuscationFile.getKeyValue(INDEX_OF_COLUMN_OBFUSCATED_VALUE_2, FIRST_OBFUSCATED_VALUE_IN_COL2));
        Assert.assertTrue(obfuscationFile.hasChanges());

        // Check persisted information in file
        Scanner scanner = null;
        try {
            scanner = new Scanner(temporaryFile);
            Assert.assertTrue(scanner.hasNext());
            final String persistedLine = scanner.nextLine();
            Assert.assertNotNull(persistedLine);
            Assert.assertArrayEquals(new String[]{ FIRST_KEY_VALUE, FIRST_OBFUSCATED_VALUE_IN_COL1, FIRST_OBFUSCATED_VALUE_IN_COL2 }, persistedLine.split(GenericObfuscationFile.FIELD_SEPARATOR));
            Assert.assertFalse(scanner.hasNext());
        } finally {
            if (scanner != null) {
                scanner.close();
            }
        }
    }

    public void test_03_addDuplicatedObfuscatedInColumn2Value() throws Throwable {
    	System.out.println("----------------- test_03_addDuplicatedObfuscatedInColumn2Value(...)");
        obfuscationFile.add(FIRST_KEY_VALUE, "xxxxxx", "xxxxxx");
    }

    public void test_04_addDuplicatedObfuscatedValue() throws Throwable {
    	System.out.println("----------------- test_04_addDuplicatedObfuscatedValue(...)");
        obfuscationFile.add("xxxxxx", FIRST_OBFUSCATED_VALUE_IN_COL1, "xxxxxx");
    }

    public void test_05_addSecondObfuscationEntry() throws Throwable, FileNotFoundException {
    	System.out.println("----------------- test_05_addSecondObfuscationEntry(...)");
        // Add a second entry
        Assert.assertFalse(obfuscationFile.containsKeyValue(SECOND_KEY_VALUE));
        obfuscationFile.add(SECOND_KEY_VALUE, SECOND_OBFUSCATED_VALUE_IN_COL1, SECOND_OBFUSCATED_VALUE_IN_COL2);
        Assert.assertTrue(obfuscationFile.containsKeyValue(SECOND_KEY_VALUE));
        Assert.assertEquals(SECOND_OBFUSCATED_VALUE_IN_COL1, obfuscationFile.getObfuscatedValue(INDEX_OF_COLUMN_OBFUSCATED_VALUE_1, SECOND_KEY_VALUE));
        Assert.assertEquals(SECOND_OBFUSCATED_VALUE_IN_COL2, obfuscationFile.getObfuscatedValue(INDEX_OF_COLUMN_OBFUSCATED_VALUE_2, SECOND_KEY_VALUE));
        Assert.assertEquals(SECOND_KEY_VALUE, obfuscationFile.getKeyValue(INDEX_OF_COLUMN_OBFUSCATED_VALUE_1, SECOND_OBFUSCATED_VALUE_IN_COL1));
        Assert.assertEquals(SECOND_KEY_VALUE, obfuscationFile.getKeyValue(INDEX_OF_COLUMN_OBFUSCATED_VALUE_2, SECOND_OBFUSCATED_VALUE_IN_COL2));
        Assert.assertTrue(obfuscationFile.hasChanges());

        // Check persisted information in file
        Scanner scanner = null;
        try {
            scanner = new Scanner(temporaryFile);
            Assert.assertTrue(scanner.hasNext());
            scanner.nextLine(); // LINE 1 
            Assert.assertTrue(scanner.hasNext());  
            final String persistedLine = scanner.nextLine(); // LINE 2
            Assert.assertNotNull(persistedLine);
            Assert.assertArrayEquals(new String[]{ SECOND_KEY_VALUE, SECOND_OBFUSCATED_VALUE_IN_COL1, SECOND_OBFUSCATED_VALUE_IN_COL2 }, persistedLine.split(GenericObfuscationFile.FIELD_SEPARATOR));
            Assert.assertFalse(scanner.hasNext());
        } finally {
            if (scanner != null) {
                scanner.close();
            }
        }
    }

    public void test_06_createObfuscationFileNonEmptyFromFile() throws Throwable {
    	System.out.println("----------------- test_06_createObfuscationFileNonEmptyFromFile(...)");
        obfuscationFile = new GenericObfuscationFile(temporaryFile, NB_OF_COLUMNS, INDEX_OF_COLUMN_KEY);
        Assert.assertNotNull(obfuscationFile);
        Assert.assertEquals(FIRST_OBFUSCATED_VALUE_IN_COL1, obfuscationFile.getObfuscatedValue(1, FIRST_KEY_VALUE));
        Assert.assertEquals(SECOND_OBFUSCATED_VALUE_IN_COL1, obfuscationFile.getObfuscatedValue(1, SECOND_KEY_VALUE));
        Assert.assertFalse(obfuscationFile.hasChanges());
    }

    public void test_07_exportToOtherFile() throws Throwable {
    	System.out.println("----------------- test_07_exportToOtherFile(...)");
        Scanner scanner = null;

        File exportedFile = new File(new File(System.getProperty("java.io.tmpdir")), String.format("other_data_file_%tF.txt", new Date()));
        final File resultFile = obfuscationFile.exportToOtherFile(exportedFile);
        try {
            // Check persisted information in file
            scanner = new Scanner(resultFile);
            Assert.assertTrue(scanner.hasNext());

            String persistedLine = scanner.nextLine();
            Assert.assertNotNull(persistedLine);
            Assert.assertArrayEquals(new String[]{ FIRST_KEY_VALUE, FIRST_OBFUSCATED_VALUE_IN_COL1, FIRST_OBFUSCATED_VALUE_IN_COL2 }, persistedLine.split(GenericObfuscationFile.FIELD_SEPARATOR));
            Assert.assertTrue(scanner.hasNext());

            persistedLine = scanner.nextLine();
            Assert.assertNotNull(persistedLine);
            Assert.assertArrayEquals(new String[]{ SECOND_KEY_VALUE, SECOND_OBFUSCATED_VALUE_IN_COL1, SECOND_OBFUSCATED_VALUE_IN_COL2 }, persistedLine.split(GenericObfuscationFile.FIELD_SEPARATOR));
            Assert.assertFalse(scanner.hasNext());
        } finally {
            if (scanner != null) {
                scanner.close();
            }
            remove(resultFile);
        }
    }
    
    private File getTempFile() throws Throwable {
		try {
			final File tempFile = File.createTempFile("file-for-testing", null);
			Assert.assertNotNull(tempFile);
			tempFile.deleteOnExit();
			return tempFile;
		} catch (final IOException ioe) {
			throw new RuntimeException(ioe);
		}
	}
	
	private void remove(final File file) {
		if (file != null && file.exists()) {
			file.delete();
		}
	}
}
