package sun.security.x509;

import java.security.Signature;
import java.security.SignatureException;
import java.security.Signer;
import java.security.NoSuchAlgorithmException;

/**
 * This class provides a binding between a Signature object and an
 * authenticated X.500 name (from an X.509 certificate chain), which
 * is needed in many public key signing applications.
 */
public final class X500Signer extends Signer
{
    private static final long serialVersionUID = -8609982645394364834L;

    /**
     * Called for each chunk of the data being signed.  That
     * is, you can present the data in many chunks, so that
     * it doesn't need to be in a single sequential buffer.
     *
     * @param buf buffer holding the next chunk of the data to be signed
     * @param offset starting point of to-be-signed data
     * @param len how many bytes of data are to be signed
     * @exception SignatureException on errors.
     */
    public void update(byte buf[], int offset, int len)
    throws SignatureException {
        sig.update (buf, offset, len);
    }

    /**
     * Produces the signature for the data processed by update().
     *
     * @exception SignatureException on errors.
     */
    public byte[] sign() throws SignatureException {
        return sig.sign();
    }

    /**
     * Returns the algorithm used to sign.
     */
    public AlgorithmId  getAlgorithmId() {
        return algid;
    }

    /**
     * Returns the name of the signing agent.
     */
    public X500Name     getSigner() {
        return agent;
    }

    /*
     * Constructs a binding between a signature and an X500 name
     * from an X.509 certificate.
     */
    // package private  ----hmmmmm ?????
    public X500Signer(Signature sig, X500Name agent) {
        if (sig == null || agent == null)
            throw new IllegalArgumentException ("null parameter");

        this.sig = sig;
        this.agent = agent;

        try {
          this.algid = AlgorithmId.getAlgorithmId(sig.getAlgorithm());

        } catch (NoSuchAlgorithmException e) {
            throw new RuntimeException("internal error! " + e.getMessage());
        }
    }

    private Signature   sig;
    private X500Name    agent;          // XXX should be X509CertChain
    private AlgorithmId algid;
}
