package com.ho.crypto.test3.encryption.keystore;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.security.KeyStore;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.cert.Certificate;

import javax.crypto.SecretKey;

import org.apache.commons.lang.StringUtils;
 

/**
 * Class of JCEKS Keystore Management
 * @author Huseyin OZVEREN
 */
public class KeyStoreManagement {

    private KeyStore keyStore;
    private String keyStoreFileName;
    private String keyStorePassword;

    public KeyStoreManagement() {
    }

    public KeyStoreManagement(String ksFileName, String ksPassword) throws Exception {
        this.keyStorePassword = ksPassword;
        this.keyStoreFileName = ksFileName;
        this.keyStore = createOrUseKeyStore(ksFileName, ksPassword);
    }

    private static KeyStore createOrUseKeyStore(String ksFileName, String ksPassword) throws Exception {
    	// KeyStore Types - see https://docs.oracle.com/javase/7/docs/technotes/guides/security/StandardNames.html#KeyStore
        final KeyStore ks = KeyStore.getInstance("JCEKS"); // or KeyStore.getDefaultType()
        InputStream input = null;
        File f = new File(ksFileName);
        if (f.exists()) {
            input = new FileInputStream(f);
            // .keystore file already exists => load it
            ks.load(input, ksPassword.toCharArray());
        } else {
            // .keystore file not created yet => create it
            ks.load(null, null);
            ks.store(new FileOutputStream(ksFileName), ksPassword.toCharArray());
        }

        return ks;
    }

    /**
     * Check if a SecretKey is stored in keystore
     * @param keyEntryName
     * @param keyEntryPassword
     * @param key
     * @return
     * @throws Exception
     */
    public boolean hasSecretKey(String keyEntryName, String keyEntryPassword, byte[] keyValue) throws Exception {
        KeyStoreManagement keyStoreManagement = new KeyStoreManagement(this.keyStoreFileName, this.keyStorePassword);
        SecretKey tmp = keyStoreManagement.getSecretKey(keyEntryName, keyEntryPassword);
        if(tmp == null){
            return false;
        } else {
            return new String(keyValue).equals(new String(tmp.getEncoded()));
        }
    }

    /**
     * Add SecretKey in keystore
     *
     * @param keyEntryName
     * @param keyEntryPassword
     * @param secretKey
     * @throws Exception
     */
    public void addSecretKey(String keyEntryName, String keyEntryPassword, SecretKey secretKey) throws Exception {
        KeyStore.SecretKeyEntry keyStoreEntry = new KeyStore.SecretKeyEntry(secretKey);
        KeyStore.PasswordProtection keyPassword = new KeyStore.PasswordProtection(keyEntryPassword.toCharArray());
        keyStore.setEntry(keyEntryName, keyStoreEntry, keyPassword);
        keyStore.store(new FileOutputStream(keyStoreFileName), keyStorePassword.toCharArray());
    }

    /**
     * Get SecretKey from keystore
     * @param keyEntryName
     * @param keyEntryPassword
     * @return
     * @throws Exception
     */
    public SecretKey getSecretKey(String keyEntryName, String keyEntryPassword) throws Exception {
		if (StringUtils.isEmpty(keyEntryName)) {
			throw new IllegalArgumentException("Key entry name should be provided");
		}
        KeyStore.PasswordProtection keyPassword = new KeyStore.PasswordProtection(keyEntryPassword.toCharArray());
        KeyStore.Entry entry = keyStore.getEntry(keyEntryName, keyPassword);
        if(entry==null){
            return null;
        } else {
            SecretKey keyFound = ((KeyStore.SecretKeyEntry) entry).getSecretKey();
            return keyFound;
        }
    }
    
    /**
     * Get PrivateKey from keystore
     * @param keyEntryName
     * @param keyEntryPassword
     * @return
     * @throws Exception
     */
    public PrivateKey getPrivateKey(String keyEntryName, final String keyEntryPassword) throws Exception {
		if (StringUtils.isEmpty(keyEntryName)) {
			throw new IllegalArgumentException("Key entry name should be provided");
		}
        KeyStore.PasswordProtection keyPassword = new KeyStore.PasswordProtection(keyEntryPassword.toCharArray());
        KeyStore.Entry entry = keyStore.getEntry(keyEntryName, keyPassword);
        if(entry==null){
            return null;
        } else {
        	PrivateKey keyFound = ((KeyStore.PrivateKeyEntry) entry).getPrivateKey();
            return keyFound;
        }
	}

    /**
     * Get Certificate from keystore
     * @param keyEntryName
     * @return
     * @throws Exception
     */
    public Certificate getCertificate(String keyEntryName) throws Exception {
		if (StringUtils.isEmpty(keyEntryName)) {
			throw new IllegalArgumentException("Key entry name should be provided");
		}
        KeyStore.Entry entry = keyStore.getEntry(keyEntryName, null);
        if(entry==null){
            return null;
        } else {
        	Certificate certificatFound = ((KeyStore.TrustedCertificateEntry) entry).getTrustedCertificate();
            if (certificatFound == null) {
                throw new IllegalArgumentException(String.format("No certificate found for alias '%s'", keyEntryName));
            }
            return certificatFound;
        }
    }

    /**
     * Get PublicKey from keystore
     * @param keyEntryName
     * @return
     * @throws Exception
     */
    public PublicKey getPublicKeyOfCertificate(String keyEntryName) throws Exception {
		if (StringUtils.isEmpty(keyEntryName)) {
			throw new IllegalArgumentException("Key entry name should be provided");
		}
        KeyStore.Entry entry = keyStore.getEntry(keyEntryName, null);
        if(entry==null){
            return null;
        } else {
        	Certificate certificatFound = ((KeyStore.TrustedCertificateEntry) entry).getTrustedCertificate();
            if (certificatFound == null) {
                throw new IllegalArgumentException(String.format("No certificate found for alias '%s'", keyEntryName));
            }
            return certificatFound.getPublicKey();
        }
    }
    
    /**
     * Add PrivateKey in keystore
     *
     * @param keyEntryName
     * @param keyEntryPassword
     * @param privateKey
     * @param certificatesChain
     * @throws Exception
     */
    public void addPrivateKey(String keyEntryName, String keyEntryPassword, PrivateKey privateKey, Certificate[] certificatesChain) throws Exception {
        KeyStore.PrivateKeyEntry keyStoreEntry = new KeyStore.PrivateKeyEntry(privateKey, certificatesChain);
        KeyStore.PasswordProtection keyPassword = new KeyStore.PasswordProtection(keyEntryPassword.toCharArray());
        keyStore.setEntry(keyEntryName, keyStoreEntry, keyPassword);
        keyStore.store(new FileOutputStream(keyStoreFileName), keyStorePassword.toCharArray());
    }

    /**
     * Add Certificate with inside PublicKey in keystore
     *
     * @param keyEntryName
     * @param trustedCert
     * @throws Exception
     */
    public void addCertificate(String keyEntryName, Certificate trustedCert) throws Exception {
        KeyStore.TrustedCertificateEntry keyStoreEntry = new KeyStore.TrustedCertificateEntry(trustedCert);
        keyStore.setEntry(keyEntryName, keyStoreEntry, null);
        keyStore.store(new FileOutputStream(keyStoreFileName), keyStorePassword.toCharArray());
    }
    
    /**
     * Check if a PrivateKey is stored in keystore
     * @param keyEntryName
     * @param keyEntryPassword
     * @param keyValue
     * @return
     * @throws Exception
     */
    public boolean hasPrivateKey(String keyEntryName, String keyEntryPassword, byte[] keyValue) throws Exception {
        KeyStoreManagement keyStoreManagement = new KeyStoreManagement(this.keyStoreFileName, this.keyStorePassword);
        PrivateKey tmp = keyStoreManagement.getPrivateKey(keyEntryName, keyEntryPassword);
        if(tmp == null){
            return false;
        } else {
            return new String(keyValue).equals(new String(tmp.getEncoded()));
        }
    }

    /**
     * Check if a Certificate is stored in keystore
     * @param keyEntryName
     * @param certificateValue
     * @return
     * @throws Exception
     */
    public boolean hasCertificate(String keyEntryName, byte[] certificateValue) throws Exception {
        KeyStoreManagement keyStoreManagement = new KeyStoreManagement(this.keyStoreFileName, this.keyStorePassword);
        Certificate tmp = keyStoreManagement.getCertificate(keyEntryName);
        if(tmp == null){
            return false;
        } else {
            return new String(certificateValue).equals(new String(tmp.getEncoded()));
        }
    }

    /**
     * Check if a PublicKey is stored in keystore
     * @param keyEntryName
     * @param keyValue
     * @return
     * @throws Exception
     */
    public boolean hasPublicKeyOfCertificate(String keyEntryName, byte[] keyValue) throws Exception {
        KeyStoreManagement keyStoreManagement = new KeyStoreManagement(this.keyStoreFileName, this.keyStorePassword);
        Certificate tmp = keyStoreManagement.getCertificate(keyEntryName);
        if(tmp == null || tmp.getPublicKey()==null){
            return false;
        } else {
        	PublicKey keyTmp = tmp.getPublicKey();
            return new String(keyValue).equals(new String(keyTmp.getEncoded()));
        }
    }
}
